/*
 * TurtleGraphicsImpl.java
 *
 * Created on February 22, 2003, 3:03 PM
 *
 * Copyright 2003 Danny Brewer
 * Anyone may run this code.
 * If you wish to modify or distribute this code, then
 *  you are granted a license to do so only under the terms
 *  of the Gnu Lesser General Public License.
 * See:  http://www.gnu.org/licenses/lgpl.html
 */

package nom.DannyBrewer.graphics;

/**
 * A mostly complete implementation of the TurtleGraphics interface.
 * All you have to write is drawLine().
 *
 * If you implement the TurtleGraphics interface, then you can copy
 *  and paste this implementation into your implementation.
 * Or you can subclass this class.
 *
 * Don't forget to set the initial turtle position to the "center",
 *  (whatever that means in your context) if at all possible.
 *
 * @author  danny brewer
 */
abstract public class TurtleGraphicsImpl implements TurtleGraphics {
    
    //----------------------------------------------------------------------
    //  Constructor
    //----------------------------------------------------------------------
    
    public TurtleGraphicsImpl() { }
    
    public TurtleGraphicsImpl( int turtleX, int turtleY ) {
        this.turtleX = turtleX;
        this.turtleY = turtleY;
    }
    
    
    //----------------------------------------------------------------------
    //  TurtleGraphics interface -- partial implementation
    //----------------------------------------------------------------------
    
    private int turtleX = 0;
    private int turtleY = 0;
    private double turtleDirection = 0.0;
    private double turtleFullCircleAngle = FULL_CIRCLE_DEGREES;
    private double turtleAngleMultiplier = FULL_CIRCLE_RADIANS / turtleFullCircleAngle;
    private boolean turtlePenDown = true;
    
    
    // Point the turtle in a certian direction specified by angle.
    // Add angle to the current angle.
    public void turn( double angle ) {
        turtleDirection = turtleDirection + angle;
    }
    
    // Point to an absolute angle.
    public void turnTo( double angle ) {
        turtleDirection = angle;
    }
    
    // Find out what direction the turtle is pointed.
    public double getTurtleDirection() {
        return turtleDirection;
    }
    
    
    // Change what units the angles are measured in.
    // If you wanted to change to measuring angles in radians,
    //  you could simply call...
    //		setFullCircleAngle( FULL_CIRCLE_RADIANS )
    //  to change angular measurements to radians.
    // To convert back to using degrees for angular measurements,
    //  simply call either...
    //		setFullCircleAngle( FULL_CIRCLE_DEGREES )
    //		setFullCircleAngle( 360 )
    // You can measure angles in any arbitrary units you wish.
    // For instance, to measure angles as fractions of a circle,
    //  just specify 1.0 as the full circle measurement.
    //		setFullCircleAngle( 1.0 )
    // A right angle would now be measured as 0.25.
    // A full turnaround would be measured as 0.5.
    // A full circle would be measured as 1.0.
    public void setFullCircleAngle( double fullCircleAngle ) {
        turtleFullCircleAngle = fullCircleAngle;
        turtleAngleMultiplier = FULL_CIRCLE_RADIANS / turtleFullCircleAngle;
    }
    public double getFullCircleAngle() {
        return turtleFullCircleAngle;
    }
    
    // Utility function
    public double convertAngleToRadians( double angle ) {
        return angle * turtleAngleMultiplier;
    }
    
    
    // Is the turtle's pen down?
    // Move the pen up or down.
    public boolean isPenDown() {
        return turtlePenDown;
    }
    public boolean isPenUp() {
        return ! turtlePenDown;
    }
    public void penDown() {
        turtlePenDown = true;
    }
    public void penUp() {
        turtlePenDown = false;
    }
    public void penDown( boolean down ) {
        turtlePenDown = down;
    }
    
    
    // Move the turtle some distance in its present direction.
    // Whether a line is drawn depends on whether the pen is down.
    public void moveForward( int distance ) {
        double angle = turtleDirection * turtleAngleMultiplier;
        int deltaX = (int) (distance * Math.cos( angle ));
        int deltaY = (int) (distance * Math.sin( angle ));
        moveBy( deltaX, deltaY );
    }
    // Move the turtle to an absolute location.
    // The turtle remains pointed in the same direction it was.
    public void moveTo( int x, int y ) {
        if( turtleX != x  ||  turtleY != y ) {
            if( turtlePenDown ) {
                drawLine( turtleX, turtleY, x, y );
            }
            turtleX = x;
            turtleY = y;
        }
    }
    // Move the turtle a relative amount.
    // The turtle remains pointed in the same direction it was.
    public void moveBy( int deltaX, int deltaY ) {
        moveTo( turtleX + deltaX, turtleY + deltaY );
    }
    
    
    // Move the turtle some distance in its present direction.
    // The pen will be put down and a line will be drawn.
    public void lineForward( int distance ) {
        turtlePenDown = true;
        moveForward( distance );
    }
    // Draw line in an absolute location.
    // The turtle remains pointed in the same direction it was.
    public void lineTo( int x, int y ) {
        turtlePenDown = true;
        moveTo( x, y );
    }
    // Draw line by a relative amount.
    // The turtle remains pointed in the same direction it was.
    public void lineBy( int deltaX, int deltaY ) {
        turtlePenDown = true;
        moveBy( deltaX, deltaY );
    }
    
    
    // Move the turtle some distance in its present direction.
    // The pen will be put up and no line will be drawn.
    public void jumpForward( int distance ) {
        turtlePenDown = false;
        moveForward( distance );
    }
    // Move the turtle to an absolute location.
    // The turtle remains pointed in the same direction it was.
    public void jumpTo( int x, int y ) {
        turtlePenDown = false;
        moveTo( x, y );
    }
    // Move the turtle a relative amount.
    // The turtle remains pointed in the same direction it was.
    public void jumpBy( int deltaX, int deltaY ) {
        turtlePenDown = false;
        moveBy( deltaX, deltaY );
    }
    
    
    public int getTurtleXPosition() {
        return turtleX;
    }
    public int getTurtleYPosition() {
        return turtleY;
    }
    
    
    
    //----------------------------------------------------------------------
    //  TurtleGraphics interface -- partial implementation
    //----------------------------------------------------------------------
    
    // Draw line.
    abstract protected void drawLine( int xStart, int yStart, int xEnd, int yEnd );
    
    
    
}
